'use strict';

angular.module('genisis').service('studyApproval', ['api', '$rootScope',
  function ($api, $rootScope) {
    var _studies = [];

    /**
     * Return the name of the study approval given the id
     * @param int id the id number of the study approval
     * @return string type name for that id
     **/
    this.toName = function (id) {

      if (!id) {
        return 'None';
      }

      var study = this.find(id);
      return (study) ? study.title : 'None';
    };

    /**
     * Load all studies or specific study approvals for a user
     * @param int uid user id
     * @return Object[] list of studies that were loaded
     **/
    this.load = function (payload) {
      return $api.retrieve('studyApprovals', payload).then(
        function (request) {

          if (request.success) {

            if (request.response instanceof Array) {
              _studies = request.response;
            } else {
              _studies = [request.response];
            }

            return _studies;
          }

          return [];
        }
      );
    };

    /**
     * Find a specific study in the list
     * Note: studies must have already been loaded
     * @param int id the study id
     * @return Object|null
     **/
    this.find = function (id) {

      if (!id) {
        return null;
      }

      //make sure it's always an integer
      id = parseInt(id, 10);

      var s = null;

      angular.forEach(_studies, function (study) {
        if (study.id === id) {
          s = study;
        }
      });

      return s;
    };

    /**
     * Return the array of all loaded studies
     * @return Object[]
     **/
    this.studies = function () {
      return _studies;
    };

    /**
     * Creating a new study approval
     * @param Object studyData the study approval data
     * @return boolean
     **/
    this.createApproval = function (studyData) {
      return $api.create('createStudyApproval', studyData).then(
        function (request) {

          if (request.success) {
            //set when this was created
            studyData.createdOn = new Date().getTime();

            //add the study approval to the list of studies
            _studies.push(studyData);
            return request.response.id;
          }

          return false;
        }
      );
    };

    /**
     * Check if the study info is valid or not
     * @param Object study data
     * @return boolean
     **/
    this.valid = function (study) {

      if (!study.title) {
        $rootScope.errors.push(
          'You must enter a study approval title.');
        return false;
      }

      if (!study.obtained) {
        $rootScope.errors.push('You must enter where the ' +
          'study approval was obtained.');
        return false;
      }

      if (!study.description) {
        $rootScope.errors.push(
          'You must enter a description of the study approval.'
        );
        return false;
      }

      return true;
    };
  }
]);